/*
 *  Font Editor
 *  -----------
 *  This program allows the editing of subfont images.
 *  It displays characters in a 16x16 grid and shows the
 *  width of each character using an adjustable red line.
 *
 *  The left mouse button draws pixels.
 *  The right mouse button adjusts the red width line.
 *
 *  Limitations:
 *
 *  1. The size of glyphs is currently hard-coded into
 *  an enum statement near the top of this file. The
 *  XSIZE and YSIZE values must be set to the pixel
 *  size of the glyphs you wish to edit, and the
 *  program recompiled. Future versions will make these
 *  variables but currently they are hard-coded. Sorry.
 *
 *  2. GraphApp currently only allows saving of GIF images,
 *  not PNG images. This is a problem because subfonts
 *  are typically stored as PNG images. So, to use this
 *  program, convert the subfont you wish to edit into
 *  GIF format using an image editor program, then edit
 *  it with this program, then convert it back to PNG.
 *  The free program gif2png may help with this task.
 *
 *  3. This program edits one subfont at a time. It creates
 *  a file specifying the widths of characters within the
 *  subfont when it saves the subfont image. This "width"
 *  file has the same base name as the subfont, but has
 *  the suffix .txt rather than .gif or .png.
 *  You must manually merge that width information into
 *  the font's master width file.
 *
 *  Example: suppose you wanted to edit the subfont
 *  "Unifont 16 italic subfont 00000000".
 *  The font will be in the fonts/ directory.
 *  Go to fonts/unifont/ and you'll see:
 *     12.txt   16.txt   16i.txt
 *     12/      16/      16i/
 *  The file "16i.txt" is the master width file for
 *  the font "Unifont 16 italic". Within the 16i/
 *  subdirectory there is a subfont named 00000000.png
 *  which contains the glyphs for subfont 0.
 *
 *  To edit that subfont, make a copy of the
 *  subfont file as a GIF image, using an editor.
 *  So we now have both 00000000.png and 00000000.gif
 *  in the 16i/ subdirectory.
 *  Next, we copy the master width file for this font
 *  into the subdirectory also, but rename it to the
 *  same base name as the subfont.
 *  For example:  copy 16i.txt 16i/00000000.txt
 *  Go to that subfont directory:   cd 16i/
 *  Now you are ready to use this program.
 *  Start it:     fontedit 00000000.gif
 */

#include <graphapp.h>
#include <str.h>

enum {
	SCALE = 10,
	XSIZE = 46,
	YSIZE = 56
};

typedef struct ImageEditorClass * ImageEditor;

struct ImageEditorClass {
	window win;	/* image editor main window */
	rgb bgcolour;	/* background colour for the window */

	menubar mbar;	/* menubar */

	menu file_menu;
	menuitem new_item, open_item, save_item;
	menuitem line_1;
	menuitem quit_item;

	menu edit_menu;
	menuitem undo_item;
	menuitem line_2;
	menuitem sort_palette;
	menuitem rotate_left;

	menu tool_menu;
	menuitem pencil, sampler, modwidth;

	control pixedit;	 /* the pixel editor */
	scrollbar vert, horiz;

	control paledit; /* the palette editor */
	control display; /* real-size display */

	field pixval;	/* colour editor */
	checkbox transparent;
	field red, green, blue;
	button add, change;

	char *filename;	/* current file */

	image prev_img;	/* the image before the current modification */
	image img;	/* the image being edited */

	int widths[256]; /* font character widths */

	int size;	/* size of one pixel when expanded */
	int x, y;	/* current x and y of top-left displayed */
	int xsize, ysize; /* page sizes */
	int changed;	/* has this image been changed? */

	int new_width;	/* pixel width of new images */
	int new_height;	/* pixel height of new images */
	int new_depth;	/* pixel depth of new images */

	window new_image_window;	/* for setting a new image's size */
	field new_width_input;	/* new image's width input field */
	field new_height_input;	/* new image's height input field */
};

/*
 *  Utilities:
 */

static int load_subfont_widths(ImageEditor editor, unsigned long sub_base)
{
	FILE *f;
	int i, r, ch, width, start, finish;
	char buf[10];
	unsigned long base;
	char name[32];

	for (i=0; i < 256; i++)
		editor->widths[i] = -1;

	/* load the font information file */
	sprintf(name, "%08lx.txt", sub_base);
	f = fopen(name, "rb");
	if (! f)
		return 0;

	base = 0UL;
	while ((ch = getc(f)) != EOF)
	{
		if ((ch == '\r') || (ch == '\n') || (ch == ' '))
			continue;
		else if (ch == '\t') {
			/* font width descriptor line */
			/* do nothing yet */
		}
		else {
			/* subfont base number */
			i = 0;
			while ((ch = buf[i++] = getc(f)) != EOF) {
				if (i == sizeof(buf)-1)
					break;
				if ((ch == '\r') || (ch == '\n'))
					break;
			}
			buf[i] = '\0';
			base = strtoul(buf, NULL, 16);
			if (base > sub_base)
				break; /* gone too far */
			continue; /* keep looking for the subfont */
		}

		/* if we are here, must be a font width descriptor line */
		if (base != sub_base) {
			/* not the subfont we were looking for */
			while ((ch = getc(f)) != EOF)
				/* discard the line */
				if ((ch == '\r') || (ch == '\n'))
					break;
			continue;
		}

		/* determine the width described by this line */
		i = 0;
		while ((ch = buf[i++] = getc(f)) != EOF) {
			if (i == sizeof(buf)-1)
				break;
			if (ch == ' ')
				break;
		}
		buf[i] = '\0';
		width = strtol(buf, NULL, 10); /* decimal integer */
		start = -1;
		finish = -2;

		/* read the width ranges */
		i = 0;
		while ((ch = getc(f)) != EOF) {
			buf[i++] = ch;
			if (i == sizeof(buf)-1)
				break;
			if (ch == '-') {
				buf[i] = '\0';
				if (start >= 0) {
					editor->widths[start] = width;
				}
				start = strtol(buf, NULL, 16);
				finish = start;
				i = 0;
			}
			if (ch == ',') {
				buf[i] = '\0';
				if (finish == start) {
					finish = strtol(buf, NULL, 16);
					for (i=start; i <= finish; i++)
						editor->widths[i] = width;
					start = -1;
				}
				else if (start >= 0) {
					editor->widths[start] = width;
					start = strtol(buf, NULL, 16);
				}
				else {
					start = strtol(buf, NULL, 16);
					finish = start - 1;
				}
				i = 0;
			}
			if ((ch == '\r') || (ch == '\n')) {
				buf[i] = '\0';
				if (finish == start) {
					finish = strtol(buf, NULL, 16);
					for (i=start; i <= finish; i++)
						editor->widths[i] = width;
				}
				else if (start >= 0) {
					editor->widths[start] = width;
					start = strtol(buf, NULL, 16);
					editor->widths[start] = width;
				}
				else {
					start = strtol(buf, NULL, 16);
					editor->widths[start] = width;
				}
				break;
			}
		}
	}

	fclose(f);
	return 1;
}

static int save_subfont_widths(ImageEditor editor, unsigned long sub_base)
{
	FILE *f;
	int i, width, start, finish, printed;
	char name[32];

	/* save the font information file */
	sprintf(name, "%08lx.txt", sub_base);
	f = fopen(name, "wb");
	if (! f)
		return 0;

	fprintf(f, "%08lX\n", sub_base);
	for (width=-1; width < 64; width++) {
		printed = 0;
		start = -1;
		finish = -2;
		for (i=0; i < 256; i++) {
			if (editor->widths[i] == width) {
				if (! printed) {
					fprintf(f, "\t%02d ", width);
					printed = 1;
				}
				if (start < 0) {
					start = finish = i;
				}
				else {
					finish++;
				}
			}
			else if (start >= 0) {
				if (printed == 2)
					fprintf(f, ",");
				if (finish > start)
					fprintf(f, "%02X-%02X", start,
						finish);
				else
					fprintf(f, "%02X", start);
				printed = 2;
				start = -1;
			}
		}
		if (start >= 0) {
			if (printed == 2)
				fprintf(f, ",");
			if (finish > start)
				fprintf(f, "%02X-%02X", start, finish);
			else
				fprintf(f, "%02X", start);
		}
		if (printed)
			fprintf(f, "\n");
	}

	fclose(f);
	return 1;
}

void reduce_palette(image img)
{
	long	i, j, length;
	int 	old_value, new_value;
	rgb 	col;
	int 	new_size;
	int *	translate;
	rgb *	new_cmap;

	if (! img)
		return;
	if (img->depth > 8)
		return;

	translate = malloc(256 * sizeof(int));

	/* Remove redundant colours: */
	length = img->width * img->height;
	for (i=0; i < 256; i++)
		translate[i] = -1;		/* colour unused */
	for (i=0; i < length; i++)
		translate[img->pixels[i]] = 1;	/* colour used */
	new_size = 0;
	for (i=0; i < 256; i++)
		if (translate[i] == 1)
			translate[i] = new_size++; /* map to this entry */

	/* Generate the new colour map: */
	new_cmap = malloc(new_size * sizeof(rgb));

	for (i=0; i < img->cmapsize; i++) {
		old_value = i;
		new_value = translate[i];
		if (new_value >= 0)
			new_cmap[new_value] = img->cmap[old_value];
	}

	/* Change the existing colour map: */
	img->cmapsize = new_size;
	for (i=0; i < new_size; i++)
		img->cmap[i] = new_cmap[i];
	free(new_cmap);

	/* Translate the pixels to the new colour map: */
	for (i=0 ; i < length; i++)
		img->pixels[i] = translate[img->pixels[i]];

	/* Clean up: */
	free(translate);
}


/*
 *  Functions:
 */

ImageEditor find_image_editor(control c)
{
	window win = parentwindow(c);
	ImageEditor editor = (ImageEditor) getdata(win);
	return editor;
}

void update_colour_editor(ImageEditor editor, int entry)
{
	rgb colour;
	rgb *palette;

	if (entry >= getpalettesize(editor->img))
		return;

	palette = getpalette(editor->img);
	colour = palette[entry];

	settext(editor->pixval, int_to_string(entry));
	if (getalpha(colour) > 0x7F)
		check(editor->transparent);
	else
		uncheck(editor->transparent);
	settext(editor->red, int_to_string(getred(colour)));
	settext(editor->green, int_to_string(getgreen(colour)));
	settext(editor->blue, int_to_string(getblue(colour)));
	draw(editor->paledit);
}

void prepare_for_change(ImageEditor editor)
{
	image img_copy;

	img_copy = copyimage(editor->img);
	del(editor->prev_img);
	editor->prev_img = img_copy;
}

void image_changed(ImageEditor editor)
{
	if (editor->changed == 0) {
		enable(editor->undo_item);
		editor->changed = 1;
	}
}

void draw_pixel_block(rgb colour, rect r)
{
	setcolour(colour);
	fillrect(r);
	if ((colour & White) == White) {	/* white */
		setcolour(LightGrey);
		drawrect(r);
	}
	if (getalpha(colour) > 0x7F) {	/* transparent */
		setcolour(Black);
		drawline(pt(r.x+r.width/2-1,r.y+r.height/2-1),
			 pt(r.x+r.width/2+2,r.y+r.height/2-1));
		drawline(pt(r.x+r.width/2,r.y+r.height/2-1),
			 pt(r.x+r.width/2,r.y+r.height/2+2));
	}
}

void redraw_pixedit(control pixedit, rect pix_rect)
{
	ImageEditor editor;
	image img;
	int x, y, width, height;
	int w, h, size;
	int maxy, maxx;
	rect r;
	byte *pixels;
	rgb *palette;
	int palsize;
	byte pixval;
	rgb colour;

	editor = find_image_editor(pixedit);
	img = editor->img;

	width = getwidth(img);
	height = getheight(img);

	size = editor->size;

	pixels = getpixels(img);
	palette = getpalette(img);
	palsize = getpalettesize(img);

	maxy = editor->ysize;
	if (maxy > height) maxy = height;
	maxx = editor->xsize;
	if (maxx > width) maxx = width;

	for (y=0; y < maxy; y++) {
	  for (x=0; x < maxx; x++) {
	    pixval = pixels[(y+editor->y)*width + (x+editor->x)];
	    r = rect(x*size+1,y*size+1,size-1,size-1);
	    if (pixval >= palsize) {	/* not in palette! */
		  setcolour(Black);
		  drawline(topleft(r), bottomright(r));
		  drawline(bottomleft(r), topright(r));
	    }
	    else {	/* correct palette entry */
		colour = palette[pixval];
		draw_pixel_block(colour, r);
	    }
	  }
	}

	/* draw font character width line */
	x = editor->x / editor->xsize;
	y = editor->y / editor->ysize;
	width = editor->widths[y*32+x];
	if (width >= 0) {
		setcolour(Red);
		fillrect(rect(width*size,1,1,maxy*size-1));
	}

	setcolour(Black);
	drawrect(pix_rect);
}

void rotate_glyph_to_left(control btn)
{
	ImageEditor editor;
	image img;
	int x, y, width, temp, pos;
	byte *pixels;

	editor = find_image_editor(btn);
	img = editor->img;

	width = getwidth(img);
	pixels = getpixels(img);

	prepare_for_change(editor);
	for (y=0; y < YSIZE; y++) {
		/* remember leftmost pixel value */
		pos = (y+editor->y)*width + editor->x;
		temp = pixels[pos];
		for (x=0; x < XSIZE-1; x++) {
			pixels[pos] = pixels[pos+1];
			pos++;
		}
		pixels[pos] = temp;
	}
	draw(editor->pixedit);
	draw(editor->display);
	image_changed(editor);
}

void handle_pixedit_drag(control pixedit, int buttons, point p)
{
	ImageEditor editor;
	image img;
	int x, y, width, height;
	int w, h, size;
	int maxy, maxx;
	rect r;
	byte *pixels;
	rgb *palette;
	int palsize;
	byte pixval;
	rgb colour;
	int index;

	editor = find_image_editor(pixedit);
	img = editor->img;

	width = getwidth(img);
	height = getheight(img);

	size = editor->size;

	pixels = getpixels(img);
	palette = getpalette(img);
	palsize = getpalettesize(img);

	maxy = editor->ysize;
	if (maxy > height) maxy = height;
	maxx = editor->xsize;
	if (maxx > width) maxx = width;

	x = (p.x - 1) / editor->size;
	y = (p.y - 1) / editor->size;
	if (x >= maxx) return;
	if (y >= maxy) return;

	pixval = pixels[(y+editor->y)*width + (x+editor->x)];
	r = rect(x*size+1,y*size+1,size-1,size-1);

	if (ischecked(editor->sampler)) {
		update_colour_editor(editor, pixval);
	}
	else if (ischecked(editor->pencil)) {
		if (buttons & LeftButton)
			index = atoi(gettext(editor->pixval));
		else if (buttons & RightButton)
			index = palsize - 1;
		if (pixval == index)
			return;
		pixval = index;
		pixels[(y+editor->y)*width + (x+editor->x)] = pixval;

		/* redraw this pixel */
		colour = palette[pixval];
		draw_pixel_block(colour, r);
		draw(editor->display);
		image_changed(editor);
	}
	else if (ischecked(editor->modwidth)) {
		int new_width = (p.x - 1 + editor->size/2) / editor->size;

		x = editor->x / editor->xsize;
		y = editor->y / editor->ysize;

		/* redraw the width line */
		width = editor->widths[y*32+x];
		if (width >= 0) {
			setcolour(White);
			fillrect(rect(width*size,1,1,maxy*size-1));
		}

		editor->widths[y*32+x] = new_width;
		if (new_width >= 0) {
			setcolour(Red);
			fillrect(rect(new_width*size,1,1,maxy*size-1));
		}
	}
}

void handle_pixedit_click(control pixedit, int buttons, point p)
{
	ImageEditor editor;

	editor = find_image_editor(pixedit);
	prepare_for_change(editor);
	handle_pixedit_drag(pixedit, buttons, p);
}

void redraw_display(control display, rect r)
{
	ImageEditor editor;
	bitmap b;

	editor = find_image_editor(display);

	drawto(display);
	r = getrect(editor->img);
	drawimage(editor->img, rect(0,0,r.width,r.height), r);

	setcolour(Black);
	drawrect(getrect(display));
}

void redraw_paledit(control paledit, rect r)
{
	ImageEditor editor;
	int i, rowsize, palsize, selected, wrong = 0;
	int height;
	rgb *palette;
	rgb colour;

	editor = find_image_editor(paledit);
	palette = getpalette(editor->img);
	palsize = getpalettesize(editor->img);

	selected = atoi(gettext(editor->pixval));
	if (selected < 0) selected = 0, wrong = 1;
	if (selected >= palsize) selected = palsize - 1, wrong = 1;
	if (wrong)
		update_colour_editor(editor, selected);

	rowsize = (r.width - 2) / SCALE;
	height = (r.height - 2) / SCALE;

	for (i=0; i < palsize; i++) {
		colour = palette[i];
		r = rect(2+SCALE*(i%rowsize),2+SCALE*(i/rowsize),8,8);
		draw_pixel_block(colour, r);
		if (i == selected)
			setcolour(Black);
		else
			setcolour(White);
		drawrect(insetr(r,-1));
	}
	setcolour(White);
	for (; i < rowsize * height; i++) {
		r = rect(2+SCALE*(i%rowsize),2+SCALE*(i/rowsize),8,8);
		fillrect(insetr(r,-1));
	}
	setcolour(Black);
	drawrect(getrect(paledit));
}

void handle_paledit_click(control paledit, int buttons, point p)
{
	ImageEditor editor;
	int entry, rowsize, palsize;
	rgb *palette;
	rect r;

	r = insetr(getrect(paledit),1);
	if (! ptinr(p,r)) return;
	p = pt(p.x-r.x,p.y-r.y);
	rowsize = r.width / SCALE;
	if (p.x > rowsize*SCALE) return;

	editor = find_image_editor(paledit);
	palette = getpalette(editor->img);
	palsize = getpalettesize(editor->img);

	entry = (p.x / SCALE) + (p.y / SCALE) * rowsize;
	if (entry >= palsize)
		return;

	update_colour_editor(editor, entry);
}

int find_colour_component(field which)
{
	int wrong = 0;
	int value;

	value = atoi(gettext(which));
	if (value < 0)   value = 0, wrong = 1;
	if (value > 255) value = 255, wrong = 1;
	if (wrong) settext(which, int_to_string(value));
	return value;
}

void add_colour(button btn)
{
	ImageEditor editor;
	int i, t, r, g, b;
	image img;
	rgb *palette;
	rgb palette2[256];
	int palsize;
	rgb colour;

	editor = find_image_editor(btn);
	img = editor->img;
	palette = getpalette(img);
	palsize = getpalettesize(img);
	if (palsize == 2<<getdepth(img)) {
		askok("Sorry, the palette is full.");
		return;
	}
	for (i=0; i < palsize; i++)
		palette2[i] = palette[i];

	if (ischecked(editor->transparent)) t = 0xFF;
	else t = 0x00;

	r = find_colour_component(editor->red);
	g = find_colour_component(editor->green);
	b = find_colour_component(editor->blue);

	colour = rgb(r,g,b) | (((rgb)t)<<24);
	palette2[palsize] = colour;

	prepare_for_change(editor);
	setpalette(img, palsize+1, palette2);
	draw(editor->paledit);
	draw(editor->display);
	image_changed(editor);
}

void change_colour(button btn)
{
	ImageEditor editor;
	int i, t, r, g, b;
	image img;
	rgb *palette;
	rgb palette2[256];
	int palsize;
	rgb colour;

	editor = find_image_editor(btn);
	img = editor->img;
	palette = getpalette(img);
	palsize = getpalettesize(img);

	for (i=0; i < palsize; i++)
		palette2[i] = palette[i];

	if (ischecked(editor->transparent)) t = 0xFF;
	else t = 0x00;

	r = find_colour_component(editor->red);
	g = find_colour_component(editor->green);
	b = find_colour_component(editor->blue);

	i = atoi(gettext(editor->pixval));
	if (i < 0) return;
	if (i >= palsize) return;

	colour = rgb(r,g,b) | (((rgb)t)<<24);
	palette2[i] = colour;

	prepare_for_change(editor);
	setpalette(img, palsize, palette2);
	draw(editor->paledit);
	draw(editor->pixedit);
	draw(editor->display);
	image_changed(editor);
}

void update_scrollbars(ImageEditor editor)
{
	int max;
	rect r = getrect(editor->pixedit);

	editor->ysize = (r.height - 2) / editor->size;
	if (editor->ysize > getheight(editor->img))
		editor->ysize = getheight(editor->img);
	max = getheight(editor->img) - editor->ysize;
	changescrollbar(editor->vert, editor->y, max/editor->ysize,1);

	editor->xsize = (r.width - 2) / editor->size;
	if (editor->xsize > getwidth(editor->img))
		editor->xsize = getwidth(editor->img);
	max = getwidth(editor->img) - editor->xsize;
	changescrollbar(editor->horiz, editor->x, max/editor->xsize,1);
}

void update_editor(ImageEditor editor)
{
	update_scrollbars(editor);
	redraw(editor->pixedit);
	redraw(editor->display);
	redraw(editor->paledit);
}

void reset_editor(ImageEditor editor)
{
	editor->x = editor->y = 0;
	update_colour_editor(editor, 0);
	update_editor(editor);
}

void move_x(scrollbar s, int value)
{
	ImageEditor editor;

	editor = find_image_editor(s);

	value = value * editor->xsize;

	if (editor->x != value) {
		editor->x = value;
		redraw(editor->pixedit);
	}
}

void move_y(scrollbar s, int value)
{
	ImageEditor editor;

	editor = find_image_editor(s);

	value = value * editor->ysize;

	if (editor->y != value) {
		editor->y = value;
		redraw(editor->pixedit);
	}
}

int save_image(ImageEditor editor, char *filename)
{
	unsigned long sub_base;

	if (! filename) /* cancel */
		return 0;
	saveimage(editor->img, filename);

	/* save subfont widths file */
	sub_base = strtoul(strchr(filename, '0'), NULL, 16);
	save_subfont_widths(editor, sub_base);

	/* change the editor filename */
	if (filename != editor->filename) {
		if (editor->filename)
			del_string(editor->filename);
		editor->filename = new_string(filename);
		settext(editor->win, filename);
	}

	/* change the settings */
	editor->changed = 0;
	disable(editor->undo_item);
	prepare_for_change(editor);

	return 1;
}

int open_image(ImageEditor editor, char *filename)
{
	image img;
	unsigned long sub_base;

	img = loadimage(filename);
	if (! img) {
		askok("Unable to load image.");
		return 0;
	}

	/* load subfont widths file */
	sub_base = strtoul(strchr(filename, '0'), NULL, 16);
	load_subfont_widths(editor, sub_base);

	/* delete the old image from memory */
	del(editor->img);

	/* change the editor filename */
	if (editor->filename)
		del_string(editor->filename);
	editor->filename = new_string(filename);
	settext(editor->win, filename);

	/* change the settings */
	editor->img = img;
	editor->changed = 0;
	disable(editor->undo_item);
	prepare_for_change(editor);

	/* redraw the various panels */
	reset_editor(editor);

	return 1;
}

int new_image(ImageEditor editor)
{
	int i, x, y, width, height;
	image img;
	byte *pixels;
	rgb palette[5] = {Black, White, Red, Green, Blue};
	int palsize = 5;

	width = editor->new_width;
	height = editor->new_height;

	img = newimage(width, height, editor->new_depth);
	if (! img) {
		askok("Insufficient memory to create a new image.");
		return 0;
	}
	pixels = getpixels(img);

	for (y=0; y < height; y++)
	  for (x=0; x < width; x++)
		pixels[y*width + x] = 1;
	setpalette(img, palsize, palette);

	/* delete the old image from memory */
	del(editor->img);

	/* change the editor filename */
	if (editor->filename)
		del_string(editor->filename);
	editor->filename = NULL;
	settext(editor->win, "untitled");

	/* change the settings */
	editor->img = img;
	editor->changed = 0;
	disable(editor->undo_item);
	prepare_for_change(editor);

	/* redraw the various panels */
	reset_editor(editor);

	return 1;
}

int ask_save_changed_image(ImageEditor editor)
{
	int result;
	char *savename;

	/* save existing modified image */
	if (editor->changed) {
		result = askyesnocancel("Image has changed. Save changes?");
		if (result == CANCEL) /* cancel */
			return CANCEL;
		if (result == YES) { /* save changes */
			savename = askfilesave("Save As:", editor->filename);
			if (! savename) /* cancel */
				return CANCEL;
			saveimage(editor->img, savename);
			editor->changed = 0;
			disable(editor->undo_item);
			prepare_for_change(editor);
		}
	}
	return YES;
}

void set_image_size(button b)
{
	ImageEditor editor = find_image_editor(b);
	editor->new_width = abs(atoi(gettext(editor->new_width_input)));
	editor->new_height = abs(atoi(gettext(editor->new_height_input)));
	new_image(editor);
	hide(parentwindow(b));
}

void ask_new_image_size(ImageEditor editor)
{
	int h;
	char buf[30];

	if (! editor->new_image_window) {
		h = getheight(SystemFont) + 5;

		editor->new_image_window = newwindow("New Image Size",
			rect(50,50,200,h+h+h+70),Titlebar+Modal);
		setdata(editor->new_image_window, editor);
		newlabel("Pixel width:", rect(10,10,120,h+10), AlignLeft);
		newlabel("Pixel height:", rect(10,h+30,120,h+10),AlignLeft);
		editor->new_width_input = newfield("", rect(135,10,50,h+10));
		editor->new_height_input = newfield("", rect(135,h+30,50,h+10));
		newbutton("Okay", rect(60,h+h+50,80,h+10), set_image_size);
	}
	sprintf(buf, "%d", editor->new_width);
	settext(editor->new_width_input, buf);
	sprintf(buf, "%d", editor->new_height);
	settext(editor->new_height_input, buf);
	show(editor->new_image_window);
}

void ask_new_image(menuitem mi)
{
	char *filename;
	ImageEditor editor;

	editor = find_image_editor(mi);

	if (ask_save_changed_image(editor) == CANCEL)
		return;
	ask_new_image_size(editor);
}

void ask_open_image(menuitem mi)
{
	char *filename;
	ImageEditor editor;

	editor = find_image_editor(mi);

	if (ask_save_changed_image(editor) == CANCEL)
		return;
	filename = askfilename("Open File:", "");
	if (filename)
		open_image(editor, filename);
}

void do_save_image(menuitem mi)
{
	char *filename;
	ImageEditor editor;

	editor = find_image_editor(mi);

	if (editor->filename)
		filename = editor->filename;
	else
		filename = askfilesave("Save As:", editor->filename);

	if (filename)
		save_image(editor, filename);
}

void ask_save_image(menuitem mi)
{
	char *filename;
	ImageEditor editor;

	editor = find_image_editor(mi);

	filename = askfilesave("Save As:", editor->filename);
	if (filename)
		save_image(editor, filename);
}

void do_quit_program(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	if (ask_save_changed_image(editor) == CANCEL)
		return;
	exitapp();
}

void use_pencil(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	check(editor->pencil);
	uncheck(editor->sampler);
	uncheck(editor->modwidth);
}

void use_sampler(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	uncheck(editor->pencil);
	check(editor->sampler);
	uncheck(editor->modwidth);
}

void use_modwidth(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	uncheck(editor->pencil);
	uncheck(editor->sampler);
	check(editor->modwidth);
}

void do_undo(menuitem mi)
{
	ImageEditor editor;
	image new_img;

	editor = find_image_editor(mi);
	if (editor->changed) {
		new_img = copyimage(editor->prev_img);
		del(editor->img);
		editor->img = new_img;
		update_editor(editor);
	}
}

void do_reduce_palette(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	prepare_for_change(editor);
	reduce_palette(editor->img);
	reset_editor(editor);
	image_changed(editor);
}

void do_sort_palette(menuitem mi)
{
	ImageEditor editor;

	editor = find_image_editor(mi);
	prepare_for_change(editor);
	sortpalette(editor->img);
	reset_editor(editor);
	image_changed(editor);
}

ImageEditor create_image_editor()
{
	ImageEditor editor = NULL;
	rect r;
	int x, h, minw, maxw;

	editor = (ImageEditor) malloc(sizeof(struct ImageEditorClass));
	if (! editor)
		return editor;

	editor->filename = NULL;
	editor->x = editor->y = 0;
	editor->size = SCALE;
	editor->changed = 0;
	editor->new_width  = 32;
	editor->new_height = 32;
	editor->new_depth  = 8;

	editor->bgcolour = LightGrey;

	r = rect(50,50,560,450);
	editor->win = newwindow("Font Editor", r, StandardWindow);
	setdata(editor->win, editor);
	setclose(editor->win, do_quit_program);

	editor->file_menu = newmenu("File");
	editor->new_item = newmenuitem("New...", 'N', ask_new_image);
	editor->open_item = newmenuitem("Open...", 'O', ask_open_image);
	editor->save_item = newmenuitem("Save", 'S', do_save_image);
	editor->save_item = newmenuitem("Save As...", 'A', ask_save_image);
	editor->line_1 = newmenuitem("-", 0, NULL);
	editor->quit_item = newmenuitem("Exit", 'Q', do_quit_program);

	editor->file_menu = newmenu("Edit");
	editor->undo_item = newmenuitem("Undo", 'Z', do_undo);
	disable(editor->undo_item);
	editor->line_2 = newmenuitem("-", 0, NULL);
	editor->sort_palette = newmenuitem("Reduce Palette", 0, do_reduce_palette);
	editor->sort_palette = newmenuitem("Sort Palette", 0, do_sort_palette);
	editor->line_2 = newmenuitem("-", 0, NULL);
	editor->rotate_left = newmenuitem("Rotate Glyph Left", 0, rotate_glyph_to_left);


	editor->tool_menu = newmenu("Tools");
	editor->pencil = newmenuitem("Pencil", 'P', use_pencil);
	check(editor->pencil);
	editor->sampler = newmenuitem("Colour Sampler", 'C', use_sampler);
	editor->modwidth = newmenuitem("Modify Width", 'M', use_modwidth);

	r = rect(10,10,2+XSIZE*SCALE,2+YSIZE*SCALE);
	editor->pixedit = newcontrol("Pixel Editor", r);
	setredraw(editor->pixedit, redraw_pixedit);
	setmousedown(editor->pixedit, handle_pixedit_click);
	setmousedrag(editor->pixedit, handle_pixedit_drag);

	editor->ysize = YSIZE;
	editor->vert = newscrollbar(rect(r.x+r.width+2,r.y,16,r.height),
			0, YSIZE, move_y);
	editor->xsize = XSIZE;
	editor->horiz = newscrollbar(rect(r.x,r.y+r.height+2,r.width,16),
			0, XSIZE, move_x);

	x = r.x + r.width + 30;
	h = getheight(SystemFont);
	maxw = strwidth(SystemFont, "<<Green>>");
	minw = strwidth(SystemFont, "456789") + 5;
	if (maxw < minw + 5) maxw = minw + 5;

	r = rect(x,r.y,150,h);
	setbackground(newlabel("Colour:", r, AlignLeft), editor->bgcolour);

	r = rect(x,r.y+r.height+5,minw,h+10);
	editor->pixval = newfield("0", r);
	disable(editor->pixval);
	r.x = r.x + maxw + 5;
	r.height = r.height + 5;

	r.width = strwidth(SystemFont, "Transparent ") + 30;
	editor->transparent = newcheckbox("Transparent", r, NULL);
	setbackground(editor->transparent, editor->bgcolour);

	r = rect(x, r.y+r.height+5, maxw, h);

	setbackground(newlabel("Red", r, AlignLeft), editor->bgcolour);
	editor->red = newfield("0", rect(r.x,r.y+h+5,minw,h+10));
	r.x = r.x + r.width + 5;

	setbackground(newlabel("Green", r, AlignLeft), editor->bgcolour);
	editor->green = newfield("0", rect(r.x,r.y+h+5,minw,h+10));
 	r.x = r.x + r.width + 5;

	setbackground(newlabel("Blue", r, AlignLeft), editor->bgcolour);
	editor->blue = newfield("0", rect(r.x,r.y+h+5,minw,h+10));

	r = rect(x,r.y + r.height + h + 20, 76, h+5);
	editor->add = newbutton("Add", r, add_colour);
	r.x = r.x + r.width + 10;
	editor->change = newbutton("Change", r, change_colour);

	r = rect(x,r.y + r.height + 5, 150, h);
	setbackground(newlabel("Palette:", r, AlignLeft), editor->bgcolour);

	r = rect(x,r.y+r.height+5,162,162);
	editor->paledit = newcontrol("Palette Editor", r);
	setredraw(editor->paledit, redraw_paledit);
	setmouseup(editor->paledit, handle_paledit_click);

	r = rect(x,r.y+r.height+10,66,66);
	editor->display = newcontrol("Display", r);
	setredraw(editor->display, redraw_display);

	setbackground(editor->win, editor->bgcolour);

	editor->prev_img = NULL;
	editor->img = NULL;
	editor->new_image_window = NULL;

	if (! new_image(editor))
		return NULL;
	update_scrollbars(editor);
	prepare_for_change(editor);

	return editor;
}

void show_image_editor(ImageEditor editor)
{
	show(editor->win);
}

int main(int argc, char *argv[])
{
	ImageEditor editor;

	if (initapp(argc, argv) == 0)
		return 1;

	editor = create_image_editor();
	if (argv && argv[1])
		open_image(editor, argv[1]);
	show_image_editor(editor);

	mainloop();
	return 0;
}
